/* Tab 2 */

/*
	ComPort.cpp - implementacja klasy CComPort
*/

#include "ComPort.h"

/******************************************************************************/

CComPort::CComPort()	/* konstruktor */
{
/*
	Tylko inicjalizacja zmiennych. Allokacja jest w Open(...)
*/	
	hCom = NULL;

	PortOutBuf = NULL;
	PortInBuf = NULL;

	LoggingState = LOGGING_INACTIVE;
}

/*----------------------------------------------------------------------------*/

CComPort::~CComPort()	/* destruktor */
{
	if( hCom )	/* jeli wczeniej nie zamknito portu to go zamkniemy teraz */
	{
		Close();
	}


	/* zamykamy MailSlot-a, jeli by utworzony */
	if( hMailSlut )
	{
		CloseHandle( hMailSlut );	
	}
}

/******************************************************************************/

int CComPort::Open( char *aPortNamePtr, DWORD aBaudRate )
{
	static BOOL RetB;
	static COMMPROP CommProp;

	if( !aPortNamePtr )
	{
		return SERROR;
	}

/*
	Transmisja synchroniczna (czyli z czekaniem na zakoczenie operacji zapisu czy te odczytu)
*/
	hCom = CreateFile( aPortNamePtr, GENERIC_READ | GENERIC_WRITE, 0, NULL,
				OPEN_EXISTING, 0, NULL);	

	if( hCom == INVALID_HANDLE_VALUE )		/* czy udao si otworzy port ? */
	{
		hCom = NULL;
		return ERR_CREATE_FILE_COM;
	}

	Dcb.DCBlength					= sizeof( DCB );
	Dcb.BaudRate					= aBaudRate;		/* Baud Rate */
	Dcb.fBinary						= TRUE;
	Dcb.fParity						= FALSE;
	Dcb.fOutxCtsFlow			= FALSE;
	Dcb.fOutxDsrFlow			= FALSE;
	Dcb.fDtrControl				= DTR_CONTROL_DISABLE;
	Dcb.fDsrSensitivity		= FALSE;
	Dcb.fTXContinueOnXoff	= FALSE;
	Dcb.fOutX							= TRUE;					/* Xon/Xoff uywane tylko podczas nadawania */
	Dcb.fInX							= FALSE;			
	Dcb.fErrorChar				= FALSE;
	Dcb.fNull							= FALSE;
	Dcb.fRtsControl				= RTS_CONTROL_DISABLE;
	Dcb.fAbortOnError			= FALSE;				/* w przypadku wystpienia bdu kolejne dostpy nie bd blokowane */
	Dcb.XonLim						= 0;				
	Dcb.XoffLim						= 0;				
	Dcb.ByteSize					= 8;						/* 8 bitw */
	Dcb.Parity						= NOPARITY;			/* bez bitu kontroli parzystoci */
	Dcb.StopBits					= ONESTOPBIT;		/* jeden bit stopu */
	Dcb.XonChar						= XON;
	Dcb.XoffChar					= XOFF;

	if( !SetCommState(hCom, & Dcb) )
	{
		CleanUp();
		return ERR_COMM_SETTINGS;
	}

/*
	Timeout na:
	- cao ( RD_INTERV_TOUT * (1 + nr_of_bytes) )
	- interwa midzybajtowy
*/

	Timeouts.ReadIntervalTimeout					= RD_INTERV_TOUT;
	Timeouts.ReadTotalTimeoutConstant			= RD_INTERV_TOUT;
	Timeouts.ReadTotalTimeoutMultiplier		= RD_INTERV_TOUT;
	Timeouts.WriteTotalTimeoutConstant		= 0;
	Timeouts.WriteTotalTimeoutMultiplier	= 0;

	if( !SetCommTimeouts( hCom, & Timeouts ) )
	{
		CleanUp();
		return ERR_COMM_SETTINGS;
	}

/*
	adne zdarzenia zwizane z portem nie bd generowane !!!
*/
	if( !SetCommMask( hCom, EV_RXCHAR ) )
	{
		CleanUp();
		return ERR_COMM_SETTINGS;
	}


/*
	Tak na dobry pocztek - czycimy co si da (kolejki...)
*/
	RetB = PurgeComm( hCom, PURGE_TXABORT | PURGE_RXABORT | PURGE_RXCLEAR | PURGE_TXCLEAR );
	_ASSERTE( RetB );


/*
	Ciekawe jak czsto bdzie brakowa pamici ?
*/
	PortInBuf = (BYTE *) calloc( PORT_INPUT_BUFFER_LEN, 1 );
	if( !PortInBuf )
	{
		CleanUp();
		return ERR_ALLOC;
	}

	PortOutBuf = (BYTE *) calloc( PORT_OUTPUT_BUFFER_LEN, 1 );
	if( !PortOutBuf )
	{
		CleanUp();
		return ERR_ALLOC;
	}

/*
	Ustawiamy dugoci kolejek: wejciowej i wyjciowej.
	Wejciowa - postaramy si ustawi jak sensown warto.
	Wyjciowa - zero (brak kolejkowania danych wychodzcych).
*/
	if( !GetCommProperties(hCom, & CommProp) )
	{
		CleanUp();
		return ERR_COMM_SETTINGS;
	}
	else
	{
		DWORD RxQueueLen;

		if( !CommProp.dwMaxRxQueue )	/* 0 oznacza, e nie ma ograniczenia rozmiaru kolejki wej. */
		{
			RxQueueLen = PORT_RX_QUEUE_LEN;
		}
		else
		{
			RxQueueLen = min( PORT_RX_QUEUE_LEN, CommProp.dwMaxRxQueue );
		}

		if( !SetupComm(hCom, RxQueueLen, 0) )
		{
			CleanUp();
			return ERR_COMM_SETTINGS;
		}
	}


/*
	W przypadku BUILD-u ze zdefiniowanym _PORTLOG, przy otwarciu portu rozpoczynane bdzie logowanies
*/

#ifdef _PORTLOG
	LoggingStart( "TxRx.log" );
#endif // _PORTLOG
	
	return SOK;
}

/*----------------------------------------------------------------------------*/

int CComPort::SetBaudRate( DWORD aRate )
{
	static BOOL RetB;

	if( !hCom )
	{
		return SERROR;
	}
	
	Dcb.BaudRate = aRate;
	RetB = SetCommState( hCom, & Dcb );		/* zmieniamy jedno pole struktury DCB  - BaudRate */
	if( RetB )
	{
		return SOK;
	}
	else
	{
		return SERROR;
	}
}

/*----------------------------------------------------------------------------*/

int CComPort::Write( DWORD aNrOfBytes, BYTE *aDataPtr )
{
	static BOOL RetB;
	static DWORD BytesWritten;

	if( (hCom == NULL) || (aDataPtr == NULL) )
	{
		return SERROR;
	}


	RetB = WriteFile( hCom, aDataPtr, aNrOfBytes, & BytesWritten, NULL );	/* wysyamy */
	if( (!RetB) || (BytesWritten != aNrOfBytes) )
	{
		return ERR_WRITE_FILE_COM;
	}

	if( LoggingState == LOGGING_ACTIVE )
	{
		return LogUpdate( LOG_WRITE, BytesWritten, aDataPtr );	/* i ewentualnie dopisujemy do logu */
	}

	return SOK;
}

/*----------------------------------------------------------------------------*/

int CComPort::Read( DWORD aNrOfBytes, DWORD *aNrOfBytesReadPtr, BYTE *aDataPtr )
{
	static DWORD NrOfBytesRead;

	if( (hCom == NULL) || (aNrOfBytesReadPtr == NULL) || (aDataPtr == NULL) )
	{
		return SERROR;
	}


/*
	Moe si zdarzy, e odczytane zostanie mniej bajtw ni dano ( nawet zero ??? )
	- wic nie bdzie porwnania aNrOfBytes i BytesRead
*/
	if( !ReadFile(hCom, aDataPtr, aNrOfBytes, & NrOfBytesRead, NULL) )	/* odczytujemy */
	{
		return ERR_READ_FILE_COM;
	}

	*aNrOfBytesReadPtr = NrOfBytesRead;

	if( LoggingState == LOGGING_ACTIVE )
	{
		return LogUpdate( LOG_READ, NrOfBytesRead, aDataPtr );	 /* i ewentualnie dopisujemy do logu */
	}

	return SOK;
}

/*----------------------------------------------------------------------------*/

int CComPort::SendSingleByte( BYTE aByte )
{
	static int RetI;

	ClearRxQueue();

	RetI = Write( 1, & aByte );
	return RetI;
}

/*----------------------------------------------------------------------------*/

void CComPort::ClearRxQueue()
{
	static BOOL RetB;
	
	RetB = PurgeComm( hCom, PURGE_RXCLEAR );		/* PurgeComm wyczyci jedynie kolejk wejciow */
	_ASSERTE( RetB );
}

/*----------------------------------------------------------------------------*/

int CComPort::WaitForAnyByte( DWORD aTimeout, BYTE *aFetchedByte )
{
	static BOOL RetB;
	static BYTE FtchdByte;
	static DWORD BytesRead;
	static COMMTIMEOUTS TempTimeouts;

/*
	Ustawiamy nowe wartoci Timeout-w 
	- w sumie tylko cakowity czas na odebranie
*/
	memset( & TempTimeouts, 0 , sizeof(TempTimeouts) );
	TempTimeouts.ReadTotalTimeoutConstant = aTimeout;
	RetB = SetCommTimeouts( hCom, & TempTimeouts );
	if( !RetB )
	{
		return ERR_COMM_SETTINGS;
	}

/*
	Odczytujemy jeden bajt 
	- troch lepsze rozwizanie ni WaitCommEvent dla operacji synchronicznych
*/
	RetB = ReadFile( hCom, & FtchdByte, 1,  & BytesRead, NULL );
	if( !RetB )
	{
		return ERR_READ_FILE_COM;
	}
	

/*
	Przywracamy normalne wartoci Timeout-w
*/
	RetB = SetCommTimeouts( hCom, & Timeouts );
	if( !RetB )
	{
		return ERR_COMM_SETTINGS;
	}

	if( !BytesRead )
	{
		return ERR_NO_RESPONSE;			/* Bd - nie doczekalimy si... */
	}
	else
	{
		/* Ewentualne zapamitanie bajtu */
		if( aFetchedByte )
		{
			*aFetchedByte = FtchdByte;
		}

/*
		Ewentualne uaktualnienie logu...
		Nic mi tu nie wstawia po tym IF-ie, bo jest RETURN !!!
*/		
		if( LoggingState == LOGGING_ACTIVE )
		{
			return LogUpdate( LOG_READ, 1, & FtchdByte );
		}
	}

	return SOK;
}

/*----------------------------------------------------------------------------*/

BOOL CComPort::WasPortOpened()
{
	return ( hCom != NULL );
}

/*----------------------------------------------------------------------------*/

int CComPort::Close()
{
	static int RetI;


	if( !hCom )
	{
		return SERROR;
	}

	RetI = SOK;

	CleanUp();

	if( LoggingState == LOGGING_ACTIVE )
	{
		RetI = LoggingStop();		/* przy zamkniciu portu wyczamy logowanie (zamykamy plik) */
	}

	return RetI;
}

/*----------------------------------------------------------------------------*/

void CComPort::CleanUp()
{
	static BOOL RetB;

	/* deallokacja buforw - o ile zdono je zaallokowa */
	if( PortOutBuf )
	{
		free( PortOutBuf );
	}

	if( PortInBuf )
	{
		free( PortInBuf );
	}

	RetB = CloseHandle( hCom );		/* waciwe zamknicie portu */
	_ASSERTE( RetB );

	hCom = NULL;
}

/******************************************************************************/

int CComPort::LoggingStart( char *aFileName )
{

	if( LoggingState == LOGGING_ACTIVE )	/* nie mona drugi raz rozpocz logowania, jeli si */
	{											/* wczeniej nie zakoczyo poprzedniej sesji */
		return SERROR;
	}

/*
	Tworzymy plik, do ktrego bd wpisywane logowane sekwencje
*/
	hLogFile = CreateFile( aFileName, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );
	
	if( hLogFile == INVALID_HANDLE_VALUE )
	{
		return ERR_CREATE_FILE;
	}

/*
	Otwieramy Mailslota do zapisu - ale tylko jeden raz. Nie bdziemy go zamyka, bo 
	mogyby by problemy z klientami.
	
	Wczeniej musi by jednak otwarty do odczytu - utworzony przez CreateMailslot
	(w programie zewntrznym).
*/

	if( ! hMailSlut )
	{
		hMailSlut = CreateFile( SNOOP_MAILSLOT_NAME,
													GENERIC_WRITE,
													FILE_SHARE_READ,
													NULL,
													OPEN_EXISTING,
													FILE_ATTRIBUTE_NORMAL,
													NULL );

		if( hMailSlut == INVALID_HANDLE_VALUE )
		{
			hMailSlut = NULL;
		}
	}

/*
	LogLineLen = 0;
	LogLastAccess = LOG_READ;
*/

	LoggingState = LOGGING_ACTIVE;	/* od tej pory wszystkie operacje R/W bd logowane */
	LogLastOperation = LOG_READ;		/* najpierw na pewno bdziemy co wysya - a inicjalizacja */
																	/* w tryb przeciwny */
	
	LogLineLen = 0;									/* na razie zero bajtw w buforze */
	
	return SOK;
}

/*----------------------------------------------------------------------------*/

int CComPort::LoggingStop()
{
	static BOOL RetB;
	static int RetI;

	if( LoggingState == LOGGING_INACTIVE )
	{
		return SERROR;
	}

	RetI = LogCreateHexAndWrite();		/* jeli co jeszcze siedziao w buforze - zapiszemy */

	LoggingState = LOGGING_INACTIVE;	/* koniec logowania */

	RetB = CloseHandle( hLogFile );		/* zamykamy plik */

	return SOK;
}

/*----------------------------------------------------------------------------*/

int CComPort::LogCreateHexAndWrite()
{
	static BOOL RetI;

	static DWORD i,
		BytesToWrite,
		BytesWritten;
	static char * DataPtr;

	DataPtr = (char *) LogStrLineBuf;

	if( LogLineLen > 0 )														/* jeli jest co w buforze... */
	{
		for( i = 0; i < LogLineLen; i++ )							/* najpierw generujemy zrzut szesnastkowy */
		{
			DataPtr += sprintf( DataPtr, "%02X ", LogByteLineBuf[i] );
		}

		for( i = LogLineLen; i < MAX_LOG_LINE; i++ )	/* produkujemy ewentualn dziur (jeli mniej ni MAX_LOG_LINE) */
		{
			DataPtr += sprintf( DataPtr, "   " );
		}

		DataPtr += sprintf( DataPtr, "   " );					/* odstp */

		for( i = 0; i < LogLineLen; i++ )							/* no i w kocu posta ASCII (lub kropka, gdy niedrukowalny) */
		{
			if( isprint(LogByteLineBuf[i]) )
			{
				DataPtr[i] = LogByteLineBuf[i];
			}
			else
			{
				DataPtr[i] = '.';
			}
		}

		DataPtr += LogLineLen;
		BytesToWrite = DataPtr - LogStrLineBuf;

		RetI = WriteToLog( (BYTE *) LogStrLineBuf, BytesToWrite, & BytesWritten, TRUE );
		if( (RetI != SOK) || (BytesWritten != BytesToWrite) )
		{
			return ERR_WRITE_FILE;
		}

		LogStrLineBuf[0] = '\n';
		RetI = WriteToLog( (BYTE *) LogStrLineBuf, 1, & BytesWritten, FALSE );
		if( (RetI != SOK) || (BytesWritten != 1) )
		{
			return ERR_WRITE_FILE;
		}
	}

	LogLineLen = 0;																/* ca zawarto bufora zapisalimy, mona wpisywa nowe rzeczy */

	return SOK;
}

/*----------------------------------------------------------------------------*/

int CComPort::LogUpdate( enum LOG_OPERATION aLogOperation, DWORD aNrOfBytes, BYTE *aDataPtr )
{
	static int RetI;

	static DWORD BytesToWrite,
		BytesWritten,
		BytesToCopy;
	
	static char OperationWr[] = "Written:",
		OperationRd[] = "Read:";

	if( !aDataPtr )
	{
		return SERROR;
	}

	if( aLogOperation != LogLastOperation )				/* gdy poprzedni dostp by innego typu */
	{
		LogCreateHexAndWrite();											/* zapiszemy to co jeszcze znajduje si w buforze */

		LogStrLineBuf[0] = '\n';
		RetI = WriteToLog( (BYTE *) LogStrLineBuf, 1, & BytesWritten, TRUE );
		if( (RetI != SOK) || (BytesWritten != 1) )
		{
			return ERR_WRITE_FILE;
		}

		if(	aLogOperation == LOG_READ )							/* nastpnie dopiszemy nagwek, mwicy jaki rodzaj dostpu */
		{
			BytesToWrite = sprintf( LogStrLineBuf, "%s", OperationRd );
		}
		else
		{
			BytesToWrite = sprintf( LogStrLineBuf, "%s", OperationWr );
		}

		RetI = WriteToLog( (BYTE *) LogStrLineBuf, BytesToWrite, & BytesWritten, TRUE );
		if( (RetI != SOK) || (BytesWritten != BytesToWrite) )
		{
			return ERR_WRITE_FILE;
		}

		LogStrLineBuf[0] = '\n';
		RetI = WriteToLog( (BYTE *) LogStrLineBuf, 1, & BytesWritten, FALSE );
		if( (RetI != SOK) || (BytesWritten != 1) )
		{
			return ERR_WRITE_FILE;
		}
	}


/*
	Dopki mamy pene wielokrotnoci MAX_LOG_LINE (po ewentualnym dopisaniu do tego, co wczeniej znajdowao si w buforze) -
	generujemy linie tekstowe i wpisujemy je do pliku
*/
	do
	{
		BytesToCopy = min( (MAX_LOG_LINE - LogLineLen), aNrOfBytes );

		memcpy( LogByteLineBuf + LogLineLen, aDataPtr, BytesToCopy );
		LogLineLen += BytesToCopy;
		aDataPtr += BytesToCopy;
		aNrOfBytes -= BytesToCopy;

		if( LogLineLen == MAX_LOG_LINE )
		{
			LogCreateHexAndWrite();
		}

	} while( aNrOfBytes > 0 );

	
	LogLastOperation = aLogOperation;							/* na koniec zapamitujemy jaka operacja zostaa dopisana do logu */
	return SOK;
}

/*----------------------------------------------------------------------------*/

int CComPort::WriteToLog( BYTE * aBufferAddr, DWORD aBytesToWrite, DWORD * aBytesWrittenPtr, BOOL aLogAlso )
{
	static BOOL RetB;

	if( (hMailSlut != NULL) && aLogAlso )
	{
		RetB = WriteFile( hMailSlut, aBufferAddr, aBytesToWrite, aBytesWrittenPtr, NULL );
	}

	RetB = WriteFile( hLogFile, aBufferAddr, aBytesToWrite, aBytesWrittenPtr, NULL ); /* dopisujemy do pliku */
	if( (! RetB) || (* aBytesWrittenPtr != aBytesToWrite) )
	{
		return ERR_WRITE_FILE;
	}

	return SOK;
}

/*----------------------------------------------------------------------------*/

int CComPort::LogFlush()
{
	if( LoggingState == LOGGING_ACTIVE )
	{
		return LogCreateHexAndWrite();
	}
	else
	{
		return SOK;
	}
}