#ifndef _posnetH_
#define _posnetH_

/*

	Nagwek w wersji: 1.0

*/


/*!
	\mainpage Biblioteka interfejsu drukarki POSNET COMBO DF
	
	Spis Treci

	- \ref sec_intro 
	- \ref sec_flow 
	- \ref sec_async
	  - \ref sec_async_events
	  .
	- \ref sec_example
	- \ref sec_extern
	- \ref linux_port
	.

	\b "SPECYFIKACJA PROTOKOU COMBO DF 1.01 / THERMAL 3.01 v 010"

	<hr>

	\section sec_intro	Wprowadzenie

	Podstawowe zaoenia, ktre zostay przyjte przy projektowaniu biblioteki dla
	drukarki POSNET COMBO DF s nastpujce:
	  - komunikacja za porednictwem RS-232, USB (VCP lub DXX)
	  - wielowtkowo i wielo-urzdzeniowo biblioteki (brak zmiennych globalnych)
	  - obsuga zdarze asynchronicznych (zmiany statusu urzdzenia)
	  - kolejkowanie rozkazw w celu przyspieszenia wymiany danych
	  - wykorzystanie z rnych jzykw programowania (konwencja wywoa __stdcall zgodna 
		z WINAPI)
	  - minimalna zaleno od systemu operacyjnego w zakresie interfejsu biblioteki -
	    praktycznie tylko 2 funkcja s typowo dedykowane dla Windows - \ref POS_GetEvent
		i \ref POS_SetMessageParams.
	  - interfejs ledzenia (\ref debugging)
	  - natywny port dla systemu Linux
	  - maksymalna zgodno API z bibliotek dla kas COMBO/NEO
	  .
    
	Ze wzgldu na to, e ilo rozkazw wykorzystywanych w komunikacji z drukark jest ogromna
	przy projektowaniu biblioteki zrezygnowano z literalnej realizacji kadego rozkazu
	jako osobnej funkcji. Grupowanie rozkazw w/g rodzaju parametrw take nie jest dobrym
	rozwizaniem. W tej sytuacji zosta zastosowany model "obiektu rozkazowego (Request)",
	wraz z zestawem funkcji umoliwiajcych ustawianie parametrw (\ref req_param). 

	Zastosowanie takiego rozwizania ma dodatkow zalet. Drobne korekty w rozkazach obsugi
	drukarki w tym dodanie nowych czy zmiana iloci i rodzaju parametrw nie wymagaj 
	zmian w bibliotece komunikacyjnej, a jedynie dostosowania aplikacji do tych zmian.

	Po wykonaniu takiego obiektu rozkazowego przez drukark (por. \ref sec_flow) wyniki 
	dostpne s w analogiczny sposb - przez zestaw funkcji pobierajcych je kolejno
	z obiektu rozkazowego.

	\section sec_flow	Przepyw informacji

	W aplikacji istniej 2 kolejki rozkazowe, pomidzy ktrymi nastpuje przepyw danych 
	jak w grafie poniej.
	\dot
	digraph flow {
		node [shape=record, fontname=Helvetica, fontsize=8];
		K0 [label="Kolejka rozkazowa"  shape="ellipse" ]
		K1 [label="Kolejka wynikow"  shape="ellipse" ]	
		NOWY -> OCZEKUJACY [label="POS_PostRequest" fontname=Helvetica fontsize=8];
		OCZEKUJACY -> K0;
		K0 -> WYSLANY [label="transmisja do drukarki" fontname=Helvetica fontsize=8];
		WYSLANY -> ZAKONCZONY [label="sekwencja wykonana\n lub blad" fontname=Helvetica fontsize=8];
		ZAKONCZONY -> K1 [label="Rozkaz" fontname=Helvetica fontsize=8]
	}
	\enddot

	Cao transmisji obsugiwana jest przez dwa niezalene wtki, jeden transmisyjny,
	drugi odbiorczy operujce na odpowiednich kolejkach rozkazw. Dodatkowo dla kadego
	urzdzenia istnieje osobny wtek wykonujcy cykliczne zapytania o status urzdzenia.

	\section sec_async	Zdarzenia asynchroniczne

	\subsection sec_async_events Zdarzenia asynchroniczne

	Drukarka POSNET COMBO DF posiada moliwo generacji 
	zdarze zmiany statusu urzdzenia w sposb asynchroniczny w 
	stosunku do wykonywanych polece
	wydawanych przez nadzorujc j aplikacj sterujco-magazynow.
	
	Do obsugi zdarze w tym trybie zosta przygotowany zestaw funkcji
	opisany w module \ref event_handling. Aplikacja gwna moe zosta 
	powiadomiona o przyjciu nowego zdarzenia
	za pomoc:
	  - zdarzenia IPC typu "Event" systemu Windows,
	  - komunikatu Windows wysanego do wskazanego okna aplikacji
	
	Aplikacja moe te cyklicznie sprawdza status urzadzenia (polling).

	UWAGA: Z natury dziaania drukarki sprawdzenie penego statusu drukarki skada si
	z dwch niezaleznych operacji, w zwizku z tym w momencie przejscia drukarki ze stanu
	poprawngo do stanu awarii mechanizmu drukujcego wystpi dwa kolejne zdarzenia 
	asynchroniczne (pierwsze wskazujce zmiane statusu drukarki, drugie statusu mechanizmu).

	UWAGA: Jeli status drukarki nie wskazuje na bd mechanizmu drukarki warto statusu
	mechanizmu jest nieistotna (niewana).

	\section sec_example	Przykadowa sekwencja polece

	Obsuga drukarki wymaga wykonania pewnego zestawu standardowych
	operacji. Poniszy opis przedstawia jedn z moliwoci na
	przykadzie prostego paragonu. Przykad jest w jzyku C++,
	dla czytelnoci pominito obsug sytuacji wyjtkowych.

	\code

	\\ Przygotowanie i otwarcie urzadzenia 
	void *hDevice=NULL;
	hDevice=POS_CreateDeviceHandle(type);
	POS_SetDebugLevel(hDevice,POSNET_DEBUG_ALL & 0xFFFFFFFE);
	POS_SetDeviceParam(hDevice,POSNET_DEV_PARAM_COMSETTINGS,(void*)"COM1,9600,8,N,1,H");
	void *hLocalDevice=POS_OpenDevice(hDevice);

	\\ Wysanie kolejnych rozkazw paragonu
	\\ Wersja z uzupenianiem parametrw pojedynczo
	hRequest = POS_CreateRequest(hLocalDevice,"trinit");
	POS_PushRequestParam(hRequest,"bm","1");
	POS_PostRequest(hRequest,POSNET_REQMODE_SPOOL);

	\\ Oczekiwanie na wykonanie rozkazu
	POS_WaitForRequestCompleted(hRequest,5000);

	\\ Sprawdzenie statusu wykonania
	if (POS_GetRequestStatus(hRequest) != POSNET_STATUS_OK) { OBSLUGA_BLEDU }

	\\ Usuniecie rozkazu
	POS_DestroyRequest(hRequest);

	\\ Nastepne polecenia, wersja z tworzeniem rozkazu z parametrami	
	hRequest = POS_CreateRequestEx(hLocalDevice,"trline","na,Buka Standardowa\nvt,0\npr,35");
	POS_PostRequest(hRequest,POSNET_REQMODE_SPOOL);
	POS_WaitForRequestCompleted(hRequest,5000);
	if (POS_GetRequestStatus(hRequest) != POSNET_STATUS_OK) { OBSLUGA_BLEDU }
	POS_DestroyRequest(hRequest);
	hRequest = POS_CreateRequestEx(hLocalDevice,"trpayment","ty,0\nre,0\nwa,35");
	POS_PostRequest(hRequest,POSNET_REQMODE_SPOOL);
	POS_WaitForRequestCompleted(hRequest,5000);
	if (POS_GetRequestStatus(hRequest) != POSNET_STATUS_OK) { OBSLUGA_BLEDU }
	POS_DestroyRequest(hRequest);
	hRequest = POS_CreateRequestEx(hLocalDevice,"trend","to,35\nfp,35");
	POS_PostRequest(hRequest,POSNET_REQMODE_SPOOL);
	POS_WaitForRequestCompleted(hRequest,5000);
	if (POS_GetRequestStatus(hRequest) != POSNET_STATUS_OK) { OBSLUGA_BLEDU }
	POS_DestroyRequest(hRequest);

	\\ zamkniecie urzadzenia
	POS_CloseDevice(hLocalDevice);
	POS_DestroyDeviceHandle(hDevice);

	\endcode

	Biblioteka posiada zestaw aplikacji demonstrujcych rne sekwencje
	polece dla drukarki.

	  -  demo_lowlevel - demonstruje przykadow realizacj pojedynczej transakcji (paragonu)
	  -  demo_requests - prezentuje rne sposoby obsugi parametrw polece oraz sposobu wysyania rozkazw i odbioru danych
	  -  demo_async2 - prezentuje realizacj obsugi statusu drukarki za porednictwem zdarze (Event) w systemie Windows lub semaforw dla systemu Linux
	  -  demo_async3 - prezentuje realizacj obsugi statusu drukarki za porednictwem funkcji typu callback
	  -  demo_all - prezentuje okoo 20 rnych przydatnych sekwencji drukarki, w tym rne warianty sprzeday, faktur VAT, formatki, raporty i ustawienia,
	  demo to obsuguje zdarzenia nieprzewidywane metod pollingu za porednictwem funkcji POS_GetPrnDeviceStatus
	  .
    Kod poszczeglnych programw demonstracyjnych zawiera w komenarzach dodatkowe informacje na temat ich dziaania.

	\section sec_extern	Wykorzystanie w jzykach programowania

	W wersji prototypowej biblioteka zostaa przetestowana w:
	  - VisualC++ .NET 2003
	  - Borland C++ Builder 4
	  - Delphi 7
	  - Visual Basic for Applications (MS Excel)
	  - Ch  (<a href='http://www.softintegration.com'>http://www.softintegration.com</a>)
	
	Poniej przedstawiono przykadowy kod dla VBA.
	\verbatim
	Private Declare Function POS_WaitForRequestCompleted Lib "posnet.dll" (ByVal H As Long, ByVal P As Long) As Long
	\endverbatim

	Podobny kod dla Delphi 7

	\verbatim
	function POS_WaitForRequestCompleted (hDevice : THandle; k: longint) : THandle; stdcall; external 'posnet.dll'
	\endverbatim

	\section linux_port Wersja dla systemu Linux

	Natywny port dla systemu Linux skada si z bibliotek:

	libposcmbth.so.1.0

	oraz 

	libptypes.so.2.0.2 (http://www.melikyan.com/ptypes/)

	Biblioteki te s zalene take od dostpnych na licencji LGPL bibliotek obsugujcych
	podsystem USB:

	libusb - w wersji 0.1.11+ dostpnej standardowo w wikszoci dystrybucji
	(http://libusb.sourceforge.net/)

	oraz 

	libftdi w wersji 0.7+ (http://www.intra2net.com/de/produkte/opensource/ftdi/index.php)
	do obsugi konwertera FTDI zastosowanego w urzdzeniu.

	UWAGA!!!  W przypadku korzystania z trybu natywnego USB (POSNET_INTERFACE_USB)
	do poprawnego dziaania drukarki niezbdne jest zablokowanie automatycznego adowania
	i usunicie jesli jest zaadowany moduu jdra ftdi_sio. W przeciwnym razie poczenie z drukark nie bdzie
	moliwe (w pliku logu zgaszany bdzie bd o kodzie -5), gdy modu ten przejmuje kontrol nad konwerterem
	FTDI tworzc wirtualny port szeregowy (/dev/ttyUSBx).

	W przypadku korzystania w trybie wirtualnego portu szeregowego, sterownik ftdi_sio jest potrzebny.
*/


/*!
	\ingroup global_defs
	\brief	Standardowy sposb obsugi bibliotek DLL
*/
/* Patch dla Borland C++ Builder'a */
#ifdef __WIN32__
#define WIN32 
#endif

#ifdef WIN32
#ifdef POSNET_EXPORTS
#define POSNET_API  __declspec(dllexport)	
#else
#define POSNET_API  __declspec(dllimport)	
#endif
#else
#define POSNET_API
#define __stdcall
#define __cdecl
#endif

extern "C"
{

	/*!
		\defgroup	global_defs Definicje globalne
		@{
		
	*/
	/*!
	\brief Definicja typu uchwytu urzdzenia.
	*/
	typedef  void*					POSNET_HANDLE;
	/*!
	\brief Definicja typu wartoci zwracanej jako status.
	*/
	typedef	 unsigned long			POSNET_STATUS;
	/*!
	\brief Definicja typu wartoci zwracanej jako stan rozkazu.
	*/
	typedef	 unsigned long			POSNET_STATE;
	/*!
	\brief Definicja minimalnego rozmianu bufora znakowego nazwy polecenia, nazwy parametru
	*/
	#define	POSNET_NAMELENGTH_MAX	16

	/*!
	\brief	Pobierz wersj biblioteki

	\return	Wersja jako liczba 32 bitowa. Liczba ta skada si z trzech czci. Najstarsze 8 bitw to gwny
	 numer wersji, nastpne 8 to podrzdny numer wersji, oststanie 16 bitw to numer kompilacji.
	*/

	POSNET_API unsigned long __stdcall POS_GetLibraryVersion();


	/*!
		@}
	*/


	/*! \defgroup debug_level	Poziomy informacyjnoci
		\ingroup debugging
	@{
	*/
	#define	POSNET_DEBUG_NONE				0x00000000  //!< Brak rejestrowania informacji.
	#define POSNET_DEBUG_ALL				0xFFFFFFFF	//!< Rejestracja wszystkich informacji we wszystkich podsystemach
	#define POSNET_SUBSYSTEM_DEVICE			0x00001000	//!< Podsystem urzdzenia
	#define POSNET_SUBSYSTEM_DEVICERS232	0x00002000  //!< Podsystem RS232
	#define	POSNET_SUBSYSTEM_DEVICEUSB		0x00008000  //!< Podsystem USB (uywa sterownika D2XX, w przypadku korzystania ze
														//!< sterownika VCP naley korzysta z trybu RS232
	#define	POSNET_SUBSYSTEM_FRAME			0x00010000  //!< Podsystem ramki - wywietla zawarto wysyanych i odbieranych ramek
														//!< w formacie hexadecymalnym - moliwo konwersji na posta binarn doczonym konwerterem
	#define POSNET_DEBUG_EXTRA				0x00800000  //!< Rejestracja szczegowych informacji o stanach rozkazw
	/*!
	@}
	*/

	/*!
		\defgroup	debugging	Ustawienie poziomu ledzenia (informacyjnoci) biblioteki
		@{
	*/

	/*!
		\brief Ustaw poziom informacyjnoci biblioteki.

		System debuggingu opiera si na tzw. "podsystemach".\n
		Definicje POSNET_DEBUG_SUBSYSTEM_* definiuj te podsystemy.\n
		S one maskami bitowymi, ktre zsumowane (OR) daja moliwo 
		jednoczesnego rejestrowania przepywu danych w rznych podsystemach.\n
		4 najmodsze bity definiuj poziom iloci wysyanych danych (stae
		POSNET_DEBUG_LEVEL_*).
		
		Ustawienie poziomu debuggingu na warto rn od 0 powoduje otwarcie
		aktualnie wybranego pliku (domylnie "POS_DBG.txt", mona ustawi przy
		wykorzystaniu \ref POS_SetDebugFileName . W pliku tym zostan zapisane
		odpowiednie informacje umoliwiajce ledzenie wykonania kodu biblioteki.
		Ponowne ustawienie poziomu debuggingu zamyka plik ledzcy.
		
		Maksymalna ilo podsystemw okrelona zostaa na 28.
		\param hGlobalDevice	Uchwyt stworzonego poczenia z urzdzeniem
		\param debugLevel	Okrelenie poziomu ledzenia
	*/
	POSNET_API	void	__stdcall POS_SetDebugLevel(POSNET_HANDLE hGlobalDevice, unsigned long debugLevel);
	/*!
		\brief	Ustawienie nazwy pliku ledzcego.

		Funkcja umoliwia zmian domylnego pliku ledzcego.
		\param hGlobalDevice	Uchwyt stworzonego poczenia z urzdzeniem
		\param fileName nowa nazwa pliku
	*/
	POSNET_API  void	__stdcall POS_SetDebugFileName(POSNET_HANDLE hGlobalDevice, const char *fileName);

	/*!
		@}
	*/



	/*! 
	*	\addtogroup dev_types Rodzaje interfejsu urzdzenia
		\ingroup basic_api

		@{
	*/

	/*!
		\brief	Podczenie przez RS232
	*/
	#define POSNET_INTERFACE_RS232	0x0001	
	/*!
		\brief	Podczenie przez USB

		Podczenie przez  USB moe by wykonane na dwa sposoby:
		-  korzystajc ze sterownika FTDI VCP (Virtual ComPort), gdzie tworzony jest dodatkowy, virtualny port szeregowy (odpowiednik RS232),
		w tym przypadku naley korzysta z trybu RS232.
		-  przy wykorzystaniu sterownika D2XX, korzysta si z trybu USB. Naley w tym przypadku poda zaprogramowany w interfejsie numer seryjny
		drukarki.
		.

	*/
	#define POSNET_INTERFACE_USB	0x0002

	/*! @}

	*/


	/*!
		\defgroup dev_params	Identyfikatory parametrw urzdzenia
		\ingroup basic_api
		@{
	*/
	
	/*!
		\brief	Parametry portu szeregowego

		Zapis/Odczyt
		
		Parametry portu przekazuje si jako cig znakw w formacie
		port,baud rate,bits,parity,stopbits,flowcontrol np. "COM1,9600,8,N,1,H"
		Flowcontrol: (N)one, (S)oftware XON/XOFF, (H)ardware RTS/CTS+DTR/DSR, (R)Hardware RTS/CTS, (D)Hardware DTR/DSR
	*/
	#define	POSNET_DEV_PARAM_COMSETTINGS	0x00020001

	/*!
		\brief Czas w [s] po jakim ma by zaniechane wysyanie ramki

		TYLKO ZAPIS

		Parametr - wskanik na wartoc long
	*/
	#define POSNET_DEV_PARAM_SENDTIMEOUT		0x00020004


	/*!
		\brief Numer seryjny drukarki do otwarcia przez typ urzdzenia \ref POSNET_INTERFACE_USB
		
		TYLKO ZAPIS

		Parametr - wskanik na cig znakw jzyka C (zakoczony 0) zawierajcy numer seryjny.

	*/
	#define POSNET_DEV_PARAM_USBSERIAL			0x00020007

	/*!

		\brief Odczyt wszystkich numerw seryjnych drukarek podczonych do komputera poprzez interfejs USB i sterownik
				FTDI - D2XX

		TYLKO ODCZYT

		Parametr - bufor na numery seryjne rozdzielone znakiem koca linii - 
					(kady numer ma max. 8znakw+2 bajty - koniec linii = 10 bajtw * max. 127 urzdze na USB = 1270)
					w zwizku z tym zalecany jest rozmiar bufora wikszy lub rwny 1271 znakw.

	*/

	#define POSNET_DEV_PARAM_LISTUSBSERIALS		0x00020008

	/*!

	\brief Dugo kolejki wysykowej, po przekroczeniu, ktrej rozkazy traktowane s jak wysyane w trybie natychmiastowym.

	TYLKO ZAPIS

	Parametr - wskanik na liczb typu unsigned long (32 bit) zawierajc dan dugo kolejki wysykowej.

	*/

	#define POSNET_DEV_PARAM_OUTQUEUELENGTH		0x00020009
	/*!
	\brief Interwa pomidzy automatycznymi odpytaniami o status drukarki

	TYLKO ZAPIS

	Parametr - wskanik na liczb typu unsigned long (32 bit) zawierajc dany czas pomidzy zapytaniami w sekundach.
	*/
	#define POSNET_DEV_PARAM_STATUSPOLLINGINTERVAL		0x0002000A


	/*!
	\brief	Pobranie uchwytu portu szeregowego

	TYLKO ODCZYT

	Parametr - Windows - wskanik na HANDLE, Linux - wskanik na liczb typu int
	*/

	#define POSNET_DEV_PARAM_FILEHANDLE		0x0002000E	


	/*!
		@}
	*/


	/*!
		\defgroup	basic_api	Obsuga podstawowa urzdzenia
		@{
	*/

	/*!
		\brief	Utworzenie uchwytu do nowego urzdzenia. 
		
		Tworzy odpowiedni obiekt i ustawia
		niebdne jego waciwoci, ale fizycznie go nie otwiera. Przed otwarciem urzdzenia 
		naley ustawi niebdne parametry, ktre s rzne w zalenoci od typu urzdzenia.
		\param	deviceType	Typ podczonego urzdzenia \ref dev_types "Typy podczenia"
		\return Uchwyt globalny utworzonego urzdzenia. Wartoc NULL zostanie zwrcona wycznie
				w przypadku braku pamici lub podania bdnego typu urzdzenia.
	*/
	POSNET_API POSNET_HANDLE	__stdcall POS_CreateDeviceHandle(unsigned long deviceType);

	/*!
		\brief	Otwrz urzdzenie.
		
		\param hGlobalDevice	Uchwyt globalny urzadzenia do otwarcia.
		\return Zwraca uchwyt lokalny do urzdzenia w kontekcie biecego wtku. NULL w przypadku bdu,
				kod bdu mona odczyta funkcj POS_GetError
	*/
	POSNET_API POSNET_HANDLE	__stdcall POS_OpenDevice(POSNET_HANDLE hGlobalDevice);		  
	/*!
		\brief	Zamknij urzdzenie. 
		
		Zamyka urzadzenie wskazywane przez uchwyt. Usuwa zawartoci kolejek
		komunikacyjnych. Wszytskie dane naley odczyta przed zamkniciem urzdzenia.

		\param hLocalDevice	Uchwyt urzadzenia do zamknicia.
	*/
	POSNET_API POSNET_STATUS	__stdcall POS_CloseDevice(POSNET_HANDLE hLocalDevice);
	/*!
		\brief	Usu istniejcy uchwyt urzdzenia i zwolnij zajmowan pami.

		Ostatecznie usuwa obiekt komunikacyjny i zwalnia pami.

		\param hGlobalDevice	Uchwyt globalny urzdzenia do usunicia. Wszystkie wtki musz najpierw zamkn urzadzenie,
								w przeciwnym wypadku funkcja zwrci bd POSNET_STATUS_BUSY.
	*/
	POSNET_API POSNET_STATUS	__stdcall POS_DestroyDeviceHandle(POSNET_HANDLE hGlobalDevice); 
	/*!
		\brief	Funkcja do ustawiania parametrw urzdzenia

		\param hDevice Uchwyt urzdzenia - zarwno lokalny jak i globalny
		\param paramCode Kod parametru por. \ref dev_params kody parametrw
		\param paramValue Wskanik lub warto parametru. Opis przy parametrach urzdzenia.

	*/
	POSNET_API POSNET_STATUS	__stdcall POS_SetDeviceParam(POSNET_HANDLE hDevice, unsigned long paramCode,void *paramValue);	
	/*!
		\brief	Funkcja do pobierania parametrw urzdzenia

		\param hDevice Uchwyt urzdzenia - zarwno lokalny jak i globalny
		\param paramCode Kod parametru por. \ref dev_params kody parametrw
		\param paramValue Wskanik lub warto parametru. Opis przy parametrach urzdzenia.

	*/
	POSNET_API POSNET_STATUS	__stdcall POS_GetDeviceParam(POSNET_HANDLE hDevice, unsigned long paramCode,void *paramValue);	



	/*!
		@}
	*/

	/*! \defgroup error_codes	Kody bdw biblioteki
		\ingroup	error_handling

		\brief	Kody bdw biblioteki. Biblioteka moe zwraca kody
				bdw zarwno wasne jak i kody bezporednio uzyskane
				z drukarki.
		@{
	*/
	
	#define POSNET_STATUS_OK						0x00000000 //!< Brak bdu
	#define POSNET_STATUS_OUTOFMEMORY				0x00000001 //!< Brak pamici
	#define POSNET_STATUS_FRAMETOOSHORT				0x00000002 //!< Za krtka ramka
	#define POSNET_STATUS_FRAMINGERROR				0x00000003 //!< Bd ramki odebranej z drukarki
	#define POSNET_STATUS_COULDNOTOPEN				0x00000005 //!< Nie mona otworzyc wskazanego urzdzenia
	#define POSNET_STATUS_CRCERROR					0x00000006 //!< Bd CRC w odebranej ramce
	#define POSNET_STATUS_IPCERROR					0x00000007 //!< Bd utworzenia obiektu IPC (Event)
	#define POSNET_STATUS_COMMERROR					0x00000008 //!< Bd komunikacji
	#define POSNET_STATUS_USBERROR					0x00000009 //!< Bd krytyczny USB - urzdzenie nie bdzie funkcjonowa poprawnie
	#define POSNET_STATUS_FTLIBIMPORTFAIL			0x0000000A //!< Nieudany import sterownika FTDI
	#define POSNET_STATUS_COULDNOTSETUPPORT			0x0000000B //!< Bd ustawienia parametrw otwieranego portu
	#define POSNET_STATUS_COULDNOTOPEN_ACCESSDENIED 0x0000000C //!< Bd otwarcia urzdzenia - dostp zabroniony
	#define POSNET_STATUS_COULDNOTOPEN_FILENOTFOUND 0x0000000D //!< Bd otwarcia urzdzenia - brak takiego pliku (urzdzenia)
	#define POSNET_STATUS_SETUP_INVALIDBAUD			0x0000000E //!< Bdne parametry portu - baudrate
	#define POSNET_STATUS_SETUP_INVALIDDATA			0x0000000F //!< Bdne parametry portu - databits
	#define POSNET_STATUS_SETUP_INVALIDPARITY		0x00000010 //!< Bdne parametry portu - parity
	#define POSNET_STATUS_SETUP_INVALIDSTOP			0x00000011 //!< Bdne parametry portu - stop bits
	#define POSNET_STATUS_SETUP_INVALIDHANDSHAKE	0x00000012 //!< Bdne parametry portu - handshake
	#define POSNET_STATUS_INVALIDSTATE				0x00000013 //!< Wydano polecenie REPEAT dla ramki znajdujcej si w niewaciwym stanie (innym ni SENT lub ACK)
	#define POSNET_STATUS_DEVICE_BUSY				0x00000014 //!< Urzdzenie zajte

	#define	POSNET_STATUS_BUSY					0x00000020 //!< Urzdzenie zajte

	#define POSNET_STATUS_ALREADY_COMPLETED		0x00010000 //!< Rozkaz ju wykonany \sa POS_CancelRequest
	#define POSNET_STATUS_EMPTY					0x00010001 //!< Brak danych w kolejce
	#define	POSNET_STATUS_INVALIDVALUE			0x00010002 //!< Bdna warto
	#define	POSNET_STATUS_TIMEOUT				0x00010003 //!< Oczekiwanie zakoczone upyniciem czasu (timeout)
	#define	POSNET_STATUS_PENDING				0x00010004 //!< Polecenie w trakcie wykonywania
	#define POSNET_STATUS_INVALIDCOMMAND		0x00010005 //!< Bdny numer polecenia
	#define POSNET_STATUS_INVALIDHANDLE			0x00010006 //!< Bdny uchwyt
	#define POSNET_STATUS_BUFFERTOOSHORT		0x00010007 //!< Przekazany bufor znakowy jest za may
	#define POSNET_STATUS_OUTOFRANGE			0x00010008 //!< Poza zakresem licznika
	#define POSNET_STATUS_INVALIDSPOOLMODE		0x00010009 //!< Bdny tryb kolejkowania
	#define POSNET_STATUS_CANCELLED				0x0001000A //!< Rozkaz anulowany

	#define POSNET_STATUS_INVALID_PARAM1		0x00010101 //!< Bdny 1 parametr polecenia
	#define POSNET_STATUS_INVALID_PARAM2		0x00010102 //!< Bdny 2 parametr polecenia
	#define POSNET_STATUS_INVALID_PARAM3		0x00010103 //!< Bdny 3 parametr polecenia
	#define POSNET_STATUS_INVALID_PARAM4		0x00010104 //!< Bdny 4 parametr polecenia
	#define POSNET_STATUS_INVALID_PARAM5		0x00010105 //!< Bdny 5 parametr polecenia
	#define POSNET_STATUS_INVALID_PARAM6		0x00010106 //!< Bdny 6 parametr polecenia

	#define	POSNET_STATUS_CASHREGBASE			0x00430000 //!< Bazowy kod bdu wykonania rozkazu drukarki [bdy 0x00430000 - 0x0043FFFF]
	#define	POSNET_STATUS_CASHREGCOMMBASE		0x00440000 //!< Bazowy kod bdu transmisji rozkazu do drukarki [bdy 0x00440000 - 0x0044FFFF]
	/*!
		@}
	*/


	/*!
		\defgroup	error_handling	Obsuga bdw
		@{
	*/

	/*!
		\brief	Podaj kod statusu zwizany z uchwytem urzdzenia
		
		Zwraca ostatnio ustawiony kod statusu zwizany z uchwytem urzdzenia. 
		Kada wywoywana funkcja oprcz zwrcenia tej wartoci jako wyniku dziaania
		(o ile nie zwraca innego rodzaju rezultatu)	ustawia wewntrzny znacznik bdu. 
		\param hLocalDevice Uchwyt lokalny urzdzenia lub uchwyt globalny urzdzenia.
		\return Kod statusu patrz \ref error_codes Kody bdw.
	*/
	POSNET_API POSNET_STATUS	__stdcall POS_GetError(POSNET_HANDLE hLocalDevice);	
	/*!
		\brief	Zwr tekstowy opis bdu

		Funkcja zwraca tekstowy opis bdu zrozumiay dla uytkownika / programisty.
		Komunikat moe by zwrcony we wskazanym jzyku (o ile przygotowano
		niezbdny plik jzykowy). W przeciwnym wypadku komunikat zwracany jest
		w domylnym jzyku biblioteki. 

		<em>UWAGA:</em> W systemie Linux pliki z tekstowymi opisami kodw bdw (pliki jzykowe
		o nazwach w postaci posnet_xx.lng mog by umieszczone w katalogu, z ktrego
		uruchomiono program (nie zalecane) lub w miejscu wskazywanym przez zmienn
		rodowiskow POSNET_LIB_SHARE_DIR (zalecane).
		W systemie WINDOWS katalog wskazany w POSNET_LIB_SHARE_DIR musi by zakoczony 
		znakiem \ rozdzielajcym katalogi np. POSNET_LIB_SHARE_DIR=c:\myapp\bin\

		\param code	Kod bdu do wyjanienia
		\param lang	2 znakowy kod jzyka wg ISO-3166
		\return Cig znakw opisujcych zdarzenia (C-string) zakoczony znakiem 0. Cig ten zwracany
				jest z wewntrznego bufora. Nie wolno go zwalnia funkcjami allokacji pamici.
	*/
	POSNET_API const char *			__stdcall POS_GetErrorString(POSNET_STATUS code, char *lang);
	/*!
		\brief	Zwr status rozkazu

		UWAGA: W celu efektywnego zarzdzania bdami (w sytuacji, gdy drukarka zwraca bdy w rznych postaciach)
		zarzdzanie bdami zostao zunifikowane - nazwa rozkazu zostaje umieszczona w polu "cm" odpowiedzi, 
		kod bdu natomiast umieszczany jest take w polu o nazwie "!error" odpowiedzi. 
		Dziki temu posta odpowiedzi jest identyczna w przypadku bdw ramki i bdw.

		\param hRequest Uchwyt rozkazu (dania wychodzcego lub odpowiedzi z drukarki)
		\return Funkcja zwraca kod bdu zwizany z rozkazem. W przypadku odebrania odpowiedzi o bdzie 
				wykonania rozkazu zwracane s kody odebrane od drukarki, s one take dostpne w polach rezultatu
				komendy w postaci zgodnej z Instrukcj Programisty.
				Kody o wartociach 0x00000000-0x0042FFFF su do przekazywania stanu rozkazu w ramach biblioteki.
				Kody o wartociach 0x00430000-0x0043FFFF wykorzystywane s w przypadku negatywnej odpowiedzi drukarki na rozkaz, po jego zakoczeniu w wyniku bdu drukarki  kod bdu drukarki zwikszony jest o warto 0x00430000.
				Kody o wartociach 0x00440000-0x0044FFFF wykorzystywane s w przypadku negatywnej odpowiedzi drukarki na rozkaz, po jego zakoczeniu w wyniku bdu transmisji  kod bdu transmisji zwikszony jest o warto 0x00440000.
		\sa sync_api "Obsuga synchroniczna drukarki"
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_GetRequestStatus(POSNET_HANDLE hRequest);

	/*!
		\brief Pobranie statusu drukarki. Drukarka jest periodycznie odpytywana o status i rezultat tej operacji jest przechowywany.
		Funkcja jest blokujca jeli wykonywane jest odpytywanie.

		\param hLocalDevice Lokalny uchwyt urzdzenia
		\param statusMode Sposb uzyskania statusu - 0 - zwr ostatni status automatyczny , 1 - odpytaj urzdzenie jeli brak statusu - blokujce!
		\param globalStatus Status urzdzenia zwracany przez rozkaz sdev (-1 jeli nie odpytano)
		\param printerStatus Status mechanizmu drukujcego  (nieistotny jeli sdev=-1)
		\return Funkcja zwraca status wykonania rozkazu, oraz wypenia pola parametrw globalStatus i printerStatus
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_GetPrnDeviceStatus(POSNET_HANDLE hLocalDevice,const char statusMode,long *globalStatus,long *printerStatus);

	/*!
		@}
	*/



	// --------------------------------------------------------------------------------------------------------
	/*! \defgroup event_types Typy zdarze
		\ingroup	event_handling
		@{
	*/
	/*!
		\brief	Wyczenie obsugi zdarze asynchronicznych - w tym trybie
		wszystkie zdarzenia s automatycznie usuwane, a aplikacja nie jest
		o nich informowana.
	*/
	#define POSNET_EVENT_DISABLE	0x0000
#ifdef _WIN32
	/*!
		\brief	Sygnalizacja otrzymania zdarzenia za pomoc komunikacji midzyprocesowej
		typu "Event" (zdarzenie) w systemie Windows.
		Po wybraniu tego typu obsugi za pomoc funkcji \ref POS_GetEvent mona
		uzyska uchwyt zdarzenia zarezerwowanego przez urzdzenie. 

		Tylko dla WIN32
	*/
	#define POSNET_EVENT_EVENT		0x0001
	/*!
		\brief	Sygnalizacja otrzymania zdarzenia za pomoc komunikatw systemu Windows.
		Po wybraniu tego trybu naley za pomoc funkcji \ref POS_SetMessageParams
		ustawi kod komunikatu i uchwyt okna do ktrego ma on by wysany. Wysyany
		komunikat ma jako lParam umieszczony uchwyt urzdzenia, ktre go wygenerowao.

		Tylko dla WIN32
	*/
	#define	POSNET_EVENT_MESSAGE	0x0002
#else
	/*!
		\brief Sygnalizacja otzrymania zdarzenia za pomoca semafora pthread

		Tylko dla Linux
	*/	
	#define POSNET_EVENT_SEMAPHORE	0x0005

#endif
	/*!
		\brief	Tryb obsugi poprzez wywoanie funkcji w momencie otrzymania
		pakietu danych asynchronicznych.
	*/
	#define POSNET_EVENT_CALLBACK	0x0004
	/*! 
		@}
	*/

	/*!
		\defgroup	event_handling	Obsuga zdarze asynchronicznych statusu drukarek
		@{
	*/


	/*!
		\brief Ustaw metod obsugi zdarze asynchronicznych 

		Drukarka ma moliwo generacji zdarze asynchronicznych w postaci zmian statusu drukarki, ktre musz by przesane do aplikacji. 
		Tego typu zdarzenia mona przesa do aplikacji w nastpujcy sposb:
		- za pomoc IPC typu "Event"
		- za pomoc komunikatw systemu Windows (Windows Messages)
		- za pomoc systemu pollingu, gdzie aplikacja odpytuje o status
		
		Funkcja suy to ustawienia trybu sygnalizacji zdarze.
		\param hLocalDevice	Uchwyt lokalny urzdzenia. Funkcja pomimo pobierania uchwytu lokalnego ma zasig
				globalny dla urzdzenia.
		\param eventMode Typ sygnalizacji zdarze definiowany przez \ref event_types "Typy zdarze"
		\return Status wykonania funkcji (POSNET_STATUS_OK w przypadku sukcesu)
	*/
	POSNET_API POSNET_STATUS	__stdcall POS_SetEventHandlingMode(POSNET_HANDLE hLocalDevice, unsigned long eventMode);
#ifdef _WIN32
	/*!
		\brief	Funkcja ustawia kod komunikatu i uchwyt okna do ktrego ma by on wysany
				przy obsudze zdarze asynchronicznych. Tylko dla Windows. Jako lParam takiego komunikatu wysyany jest
				zoony status urzdzenia (w/g wzoru: (Status urzdzenia * 65536)+status mechanizmu )
		\param hLocalDevice	Uchwyt lokalny urzdzenia. Funkcja pomimo pobierania uchwytu lokalnego ma zasig
				globalny dla urzdzenia.
		\param	messageCode	Kod komunikatu (WM_xxxx)
		\param	hWnd	Uchwyt okna

		Tylko dla WIN32
	*/
	POSNET_API POSNET_STATUS	__stdcall POS_SetMessageParams(POSNET_HANDLE hLocalDevice, UINT messageCode, HANDLE hWnd);
	/*!
		\brief	Funkcja zwraca uchwyt do zdarzenia. Obiekt zdarzenia jest zarzdzany przez
				bibliotek. Zdarzenie pracuje w trybie Manual Reset 
				patrz \sa http://msdn.microsoft.com/library/default.asp?url=/library/en-us/dllproc/base/createevent.asp MSDN.
		\param hLocalDevice	Uchwyt lokalny urzdzenia. Funkcja pomimo pobierania uchwytu lokalnego ma zasig
				globalny dla urzdzenia.
		\return	NULL w przypadku bdu, w przeciwnym przypadku uchwyt zdarzenia (Event).

		Tylko dla WIN32
	*/
	POSNET_API HANDLE			__stdcall POS_GetEvent(POSNET_HANDLE hLocalDevice);
#else
	/*!
		\brief Pobranie semafora synchronizujcego wtki pthread (typ sem_t)
		\param hLocalDevice Uchwyt lokalny urzdzenia. Funkcja pomimo pobierania uchwytu lokalnego ma zasig
				globalny dla urzdzenia.
		\return NULL w przypadku bdu, w przeciwnym wypadku wskanik na semafor.

		Tylko dla Linux
	*/
	POSNET_API void * __stdcall POS_GetSemaphore(POSNET_HANDLE hLocalDevice);

#endif
	/*!
		\brief Typ definiujcy funkcj callback dla zdarze asynchronicznych. Przyjmuje jeden parametr
		- zoony status urzdzenia (w/g wzoru: (Status urzdzenia * 65536)+status mechanizmu ) . 
	*/
	typedef void (POSNET_CALLBACK_T)(unsigned long status);

	/*!
		\brief Funkcja ustawia w bibliotece funkcj callback, ktra bdzie wywoywana po otrzymaniu zdarzenia asynchronicznego.
		\param hLocalDevice Uchwyt lokalny urzdzenia. Funkcja pomimo pobierania uchwytu lokalnego ma zasig
				globalny dla urzdzenia.
		\param callback wskanik na funkcj typu callback.
	*/
	POSNET_API POSNET_STATUS	__stdcall POS_SetCallback(POSNET_HANDLE hLocalDevice, POSNET_CALLBACK_T *callback);

	/*!
		@}
	*/
	// --------------------------------------------------------------------------------------------------------


	/* MAIN COMMAND API */

	/*!
		\defgroup	req_modes	Tryby wysyania rozkazw
		\ingroup sync_api
		@{
	*/

	#define	POSNET_REQMODE_SPOOL	0x00 //!< Tryb kolejkowania [domylny], w tym trybie rozkaz umieszczany jest na 
										 //!< kocu kolejki rozkazw do wysania, za wyjtkiem nastpujcych komend,
										 //!< ktre ze swej natury umieszczane s na samym pocztku kolejki:
										 //!< !sdev, !sprn
	#define POSNET_REQMODE_IMMEDIATE 0x01 //!< Tryb bezporedni. Podobny do \ref POSNET_REQMODE_SPOOL kolejkowania, lecz
										  //!< w przypadku, gdy w kolejce s ju jakie rozkazy zwraca bd - efektywnie
										  //!< umoliwia umieszczenie rozkazu w kolejce WYCZNIE gdy jest ona pusta,
										  //!< za wyjtkiem rozkazw !sdev i !sprn
	#define	POSNET_REQMODE_SPOOLSPECIAL	0x02 //!< Tryb kolejkowania specjalnego, w ktrym
											 //!< w kolejce odbiorczej odpowied OK (brak bdu)
											 //!< na rozkaz nie zwracajcy wyniku zostanie automatycznie
											 //!< usunita z kolejki odbiorczej. Ten tryb musi by uywany 
											 //!< jedynie w trybie odbioru z kolejki odbiorczej (nie wolno
											 //!< stosowa \ref POS_WaitForRequestCompleted POS_WaitForRequestCompleted, gdy
											 //!< nie ma gwarancji poprawnoci uchwytu urzdzenia - nie dotyczy to rozkazw
											 //!< wykonanych znajdujcych si w kolejce odbiorczej).
	#define POSNET_REQMODE_AUTOCLEAR 0x03 //!< Ten tryb powoduje automatyczne czyszczenie odpowiedzi,
										  //!< bez wzgldu na kod powrotu. W momencie wysania rozkazu w tym trybie naley uzna,
										  //!< e uchwyt rozkazu jest bdny.

	/*!
		@}
	*/
	
	/*!
		\defgroup	req_states	Stany rozkazu
		\ingroup sync_api
		@{
	*/

	#define POSNET_RSTATE_NEW			0x00019000	//!< Nowe polecenie, nie umieszczone w kolejce
	#define POSNET_RSTATE_PENDING		0x00019001	//!< Nowe polecenie, umieszczone w kolejce
	#define POSNET_RSTATE_SENT			0x00019002	//!< Polecenie wysane, nie potwierdzone
	#define	POSNET_RSTATE_COMPLETED		0x00019004	//!< Wykonanie rozkazu zakoczone
	#define	POSNET_RSTATE_ERRCOMPLETED	0x00019005	//!< Wykonanie rozkazu zakoczone bdem
	/*!
		@}
	*/


	// --------------------------------------------------------------------------------------------------------


	/*!
		\defgroup	request_base	Obsuga obiektw rozkazowych
		\ingroup sync_api

		@{
	*/

	/*!
		\brief	Funkcja tworzy nowy obiekt rozkazowy dla drukarki.

		Funkcja tworzy obiekt rozkazowy dla drukarki. Obiekt tego typu jest podstawow
		komrk transmisyjn pomidzy aplikacj a drukark. Jego zadaniem jest realizacja
		pojedynczej transakcji drukarka-aplikacja.

		Wicej informacji we wprowadzeniu - \ref sec_flow "Przepyw informacji" 

		\param	hLocalDevice	Uchwyt lokalny urzdzenia
		\param	command	Identyfikator rozkazu dla drukarki
		\return	Zwracany jest uchwyt do nowego obiektu rozkazowego lub NULL w przypadku bdu
	*/
	POSNET_API	POSNET_HANDLE	 __stdcall POS_CreateRequest(POSNET_HANDLE hLocalDevice, const char *command);
	/*!
		\brief	Funkcja tworzy nowy obiekt rozkazowy dla drukarki oraz dodaje parametry rozkazu.

		Funkcja tworzy obiekt rozkazowy dla drukarki. Obiekt tego typu jest podstawow
		komrk transmisyjn pomidzy aplikacj a drukark. Jego zadaniem jest realizacja
		pojedynczej transakcji drukarka-aplikacja.

		Wicej informacji we wprowadzeniu - \ref sec_flow "Przepyw informacji" 

		\param	hLocalDevice	Uchwyt lokalny urzdzenia
		\param	command	Identyfikator rozkazu dla drukarki
		\param  parameters Lista parametrw do dodania (zastpuje dodatkowe wywoania \ref POS_PushRequestParam) 
		format: [nazwa parametru],[wartosc]\n[nazwa parametru],[wartosc]\n....[nazwa parametru],[wartosc]
		UWAGA: Jeli warto parametru zawiera w sobie znaki nowej linii lub przecinka naley uy \ref POS_PushRequestParam aby
		doda taki parametr do rozkazu.
		\return	Zwracany jest uchwyt do nowego obiektu rozkazowego lub NULL w przypadku bdu
	*/
	POSNET_API      POSNET_HANDLE  __stdcall POS_CreateRequestEx(POSNET_HANDLE hLocalDevice, const char* command,const char *parameters);
	/*!
		\brief Funkcja suy do pobrania uchwytu urzdzenia zwizanego z rozkazem

		\param hRequest	Uchwyt rozkazu
		\return uchwyt lokalny urzdzenia lub NULL w przypadku bdu gdy nie mona ustali adresata rozkazu.
	*/
	POSNET_API POSNET_HANDLE	__stdcall POS_RequestDevice(POSNET_HANDLE hRequest);

	/*!
		\brief	Fukcja umieszcza obiekt rozkazowy w kolejce rozkazw do wykonania.

		Istniej dwa warianty wysyania rozkazw 
		  - tryb kolejkowania,
		  - tryb bezporedni.
		
		W trybie kolejkowania nowy rozkaz jest umieszczany w kolejce do wysania do drukarki i
		automatycznie wysyany w miar zwalniania si bufora drukarki. 
		\warning Dugoc kolejki jest ograniczona. Po jej przekroczeniu rozkazy traktowane s 
				 jakby byy wysyane w trybie bezporednim.
	   
	    Tryb bezporedni kontroluje, czy drukarka jest wolna i dopiero gdy jest wolna umoliwia
		umieszczenie rozkazu w buforze. W ten sposb w danej chwili do drukarki wysany jest 
		co najwyej 1 rozkaz.

		\param	hRequest	Uchwyt rozkazu
		\param	mode		Wybr trybu wysyania rozkazu \ref req_modes "Tryby wysyania rozkazw"
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_PostRequest(POSNET_HANDLE hRequest,unsigned char mode);
	/*!
		\brief	Anulowanie obiektu rozkazowego

		Funkcja anuluje wykonanie rozkazu. W zalenoci od stanu rozkazu moe to oznacza:
		  - jeli rozkaz nie zosta jeszcze wysany zostaje on usunity z kolejki rozkazw,
		  - jeli rozkaz zosta wysany do drukarki zostaje on usunity z kolejki rozkazw,
		  - jeli rozkaz by ju wykonany to nic nie jest wykonywane i zostaje zwrcony
			kod bdu \ref POSNET_STATUS_ALREADY_COMPLETED .

		W kadym przypadku za zniszczenie obiektu odpowiada aplikacja -  dla anulowanego rozkazu naley
		wywoa \ref POS_DestroyRequest .

		\param hRequest	Uchwyt rozkazu
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_CancelRequest(POSNET_HANDLE hRequest);
	/*!
		\brief	Zniszczenie obiektu rozkazowego i zwolnienie zajmowanej przeze pamici.

		Funkcja kasuje obiekt rozkazowy oraz wszystkie powizane z nim dane.
		Musi by uyta dla kadego rozkazu allokowanego poprzez \ref POS_CreateRequest POS_CreateRequest 
		ktry nie jest automatycznie usuwany w ytybach SPOOLSPECIAL i AUTOCLEAR.

		\param hRequest Uchwyt rozkazu (dania wychodzcego lub odpowiedzi z drukarki)
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_DestroyRequest(POSNET_HANDLE hRequest);

	/*!
		\brief	Wiek rozkazu - czas jaki upyn od wywoania \ref POS_PostRequest

		Funkcja ta ma na celu ustalenie wieku rozkazu i moe by przydatna w obsudze
		sytuacji nadzwyczajnych. Czas podawany jest w milisekundach.

		\param hRequest Uchwyt rozkazu (dania wychodzcego lub odpowiedzi z drukarki)
	*/
	POSNET_API	unsigned long	__stdcall POS_GetRequestAge(POSNET_HANDLE hRequest);
	/*!
		\brief	Pobierz numer polecenia, ktry zosta uyty przy tworzeniu obiektu

		\param hRequest Uchwyt rozkazu (dania wychodzcego lub odpowiedzi z drukarki)
		\param retCommand   Bufor znakowy, w ktrym zostanie umieszczona nazwa polecenia
		\return Status wykonania rozkazu
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_GetRequestCommandID(POSNET_HANDLE hRequest,char *retCommand);
	/*!
		\brief	Pobierz numer seryjny rozkazu

		\param hRequest Uchwyt rozkazu (dania wychodzcego lub odpowiedzi z drukarki)
		\return Token (numer seryjny) powizany z rozkazem
	*/
	POSNET_API	unsigned long	__stdcall POS_GetRequestCommandSerial(POSNET_HANDLE hRequest);
	/*!
		\brief	Zwr stan rozkazu

		Funkcja zwraca stan w jakim aktualnie znajduje si rozkaz.
		Wicej informacji we wprowadzeniu - \ref sec_flow "Przepyw informacji" 

		\param hRequest Uchwyt rozkazu (dania wychodzcego lub odpowiedzi z drukarki)
	*/
	POSNET_API	POSNET_STATE	__stdcall POS_GetRequestState(POSNET_HANDLE hRequest);
	/*!
		\brief	Czekaj na zakoczenie rozkazu

		Funkcja powoduje zatrzymanie (upienie) woajcej j aplikacji do momentu
		zmiany stanu rozkazu na \ref POSNET_RSTATE_COMPLETED lub POSNET_RSTATE_ERRCOMPLETED lub upynicia czasu
		wskazanego przez parametr timeout.
 
		\param hRequest	Uchwyt rozkazu
		\param	timeout Czas w ms na jaki ma aplikacja by upiona. 0 oznacza natychmiastowy 
				powrt
		\return	zwraca POSNET_STATUS_OK jeli rozkaz zosta juz ukoczony lub POSNET_STATUS_TIMEOUT 
				jeli rozkaz nie zdy si zakoczy we wskazanym czasie.
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_WaitForRequestCompleted(POSNET_HANDLE hRequest,unsigned long timeout);

	/*!
		@}
	*/

	// --------------------------------------------------------------------------------------------------------

	/*!
		\defgroup req_param	Obsuga parametrw rozkazu
		\ingroup sync_api

		\brief  W zalenoci od typu rozkazu moe on posiada odpowiedni
				ilo parametrw. Parametry te  odkadane s na list parametrw
				za pomoc kolejnych wywoa funkcji POS_PushRequestParam. Drukarka 
				COMBO DF przyjmuje parametry bez wzgldu na ich kolejno.
		@{
	*/

	/*!
		\brief	Wstaw parametr rozkazu
		\param	hRequest	Uchwyt rozkazu
		\param  param_name		Nazwa parametru reprezentowana jako cig znakw w konwencji jzyka C (zakoczony znakiem 0)
		\param	param_value		Warto parametru reprezentowana jako cig znakw w konwencji jzyka C (zakoczony znakiem 0)
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_PushRequestParam(POSNET_HANDLE hRequest,const char *param_name,const char *param_value);

	/*!
		@}
	*/
	// --------------------------------------------------------------------------------------------------------

	/*!
		\defgroup	response_handling	Analiza pl odpowiedzi
		\ingroup sync_api

		@{
	*/
	/*!
		\brief	Pobierz ilo otrzymanych pl wartoci

		Funkcja zwraca ilo pl DATA otrzymanych w odpowiedzi na rozkaz. Poprawna tylko
		wtedy, gdy sam rozkaz jest w stanie \ref POSNET_RSTATE_COMPLETED

		\param hRequest	Uchwyt zapytania/odpowiedzi
		\return Ilo otrzymanych pl odpowiedzi lub -1 w przypadku bdu
	*/
	POSNET_API	long			__stdcall POS_GetResponseValueCount(POSNET_HANDLE hRequest);

	/*!
		\brief	Pobierz warto parametru i przesu na nastpne pole

		\param	hRequest	Uchwyt zapytania/odpowiedzi
		\param retName Wskanik do bufora znakowego, w ktrym funkcja umieci nazw odebranego parametru. Dugo tego bufora MUSI by conajmniej
						POSNET_NAMELENGTH_MAX
		\param  retVal Wskanik do bufora znakowego, w ktrym funkcja umieci odebrany cig znakw. Aplikacja musi zapewni odpowiedni
						pojemno bufora, ktr przekazuje w parametrze retValLen. W przypadku zbyt krtkiego
						bufora zostanie zwrcony bd \ref POSNET_STATUS_BUFFERTOOSHORT
		\param	retValLen	Dugo bufora
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_PopResponseValue(POSNET_HANDLE hRequest,char *retName, char * retVal, unsigned long retValLen);
	/*!
		\brief	Pobierz wato parametru po nazwie

		\param	hRequest	Uchwyt zapytania/odpowiedzi
		\param paramName Wskanik do bufora znakowego, w ktrym znajduje si nazwa poszukiwanego pola odpowiedzi
		\param  retVal Wskanik do bufora znakowego, w ktrym funkcja umieci odebrany cig znakw. Aplikacja musi zapewni odpowiedni
						pojemno bufora, ktr przekazuje w parametrze retValLen. W przypadku zbyt krtkiego
						bufora zostanie zwrcony bd \ref POSNET_STATUS_BUFFERTOOSHORT
		\param	retValLen	Dugo bufora
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_GetResponseValue(POSNET_HANDLE hRequest,const char *paramName, char * retVal, unsigned long retValLen);
	/*!
		\brief	Przesu wskanik pl na pocztek listy

		Funkcja umoliwia rozpoczcie analizy pl od pocztku.

		\param	hRequest	Uchwyt zapytania/odpowiedzi
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_ResponseRewind(POSNET_HANDLE hRequest);
	
	/*!
		@}
	*/

	/*!
		\defgroup req_queue Obsuga kolejek rozkazw i wynikw
		\ingroup sync_api

		Aplikacja moe obsugiwa wysyane rozkazy w dwch trybach.
		  * W trybie "wylij-zapomnij" aplikacja tworzy rozkaz, wypenia parametrami a nastpnie
			umieszcza w kolejce rozkazw do wykonania. Jednoczenie nie przechowuje samodzielnie 
			uchwytw wysanych rozkazw. Wyniki rozkazw pobiera z kolejki wynikw za porednictwem
			\ref POS_GetNextResponse i obsuguje odpowiednio.
		  * W trybie "interaktywnym" aplikacja tworzy rozkaz, wypenia parametrami i wstawia
			do kolejki rozkazw. Jednake zapamituje uchwyt rozkazu i obserwujc stan rozkazu
			za porednictwem \ref POS_GetRequestState obsuguje go w momencie zakoczenia.
		
		Tryb "wylij-zapomnij" jest bardziej odpowiedni dla operacji masowych (np. sprawdzanie bazy towarw)
		gdy umoliwia efektywniejsze (szybsze) wykonywanie rozkazw, ktrych wynik jest
		mniej istotny.

		Tryb "interaktywny" wskazany jest dla rozkazw, ktrych wynik jest w danej chwili niezbdny dla
		dalszej pracy aplikacji. W tym celu wprowadzony jest te rozkaz \ref POS_WaitForRequestCompleted
		ktry umozliwia synchronizowane oczekiwanie na zakoczenie wykonania rozkazu przez drukark.

		\warning	Nie ma moliwoci przegldania kolejki rozkazw oczekujcych/wysanych. Jest to spowodowane 
		ich asynchronicznym w stosunku do apliacki wykonywaniem i zawarto tej kolejki moe si
		zmienia w trakcie wykonywania zapyta przegldajcych (rozkazy mog znika z kolejki mimo
		i aplikacja pobraa je jeszcze z kolejki oczekujcych).

		@{
	*/

	/*!
		\brief	Pobierz z kolejki odpowiedzi (rozkazw zakoczonych) nastpny dostpny wynik.

		Z kolejki odpowiedzi zostaje pobrany (i usunity) nastpny wynik. Pobrany wynik naley 
		zniszczy za pomoc \ref POS_DestroyRequest. Pobierane s wycznie wyniki otrzymane na
		rozkazy wysane za porednictwem biecego kontekstu urzdzenia (hLocalDevice). Rozkazy 
		wysane za pomoc innych kontekstw tego samego urzdzenia s pomijane.
		UWAGA: Uchwyt wyniku jest tosamy z uchwytem rozkazu, ktry go spowodowa - oznacza to,
		e niszczenie za pomoc \ref POS_DestroyRequest rozkazu jest rwnoznaczne ze zniszczeniem
		odpowiedzi i odwrotnie.

		Funkcja jest nieblokujca.

		\param	hLocalDevice	Uchwyt lokalny urzdzenia
		\return	Uchwyt do odpowiedzi lub NULL w przypadku bdu lub braku nastpnej odpowiedzi
	*/
	POSNET_API	POSNET_HANDLE	__stdcall POS_GetNextResponse(POSNET_HANDLE hLocalDevice);
	/*!
		\brief	Pobierz ilo wynikw w kolejce wynikw

		\param hLocalDevice	Uchwyt lokalny urzdzenia
		\return Ilo wynikw dla biecego kontekstu urzdzenia.
	*/
	POSNET_API	unsigned long	__stdcall POS_GetResponseCount(POSNET_HANDLE hLocalDevice);
	/*!
		\brief	Pobierz ilo rozkazw oczekujcych w kolejce rozkazw wysanych przez biecy kontekst
				urzdzenia lub wszystkich rozkazw oczekujcych w kolejce.

		\param hLocalDevice	Uchwyt lokalny urzdzenia
		\param globalnie Jeli parametr ten jest rny od 0 zwracana warto podaje ilo wszystkich rozkazw 
				oczekujcych w kolejce.
	*/
	POSNET_API	unsigned long	__stdcall POS_GetRequestCount(POSNET_HANDLE hLocalDevice,unsigned char globalnie=0);

	/*!
		\brief	Pobierz statystyki kolejek. Statystyki s zwracane w kontekcie caego urzdzenia.

		\param hLocalDevice	Uchwyt lokalny urzdzenia
		\param sent		Do tej zmiennej wpisywana jest ilo wszystkich wysanych do drukarki
						rozkazw
		\param completed	Do tej zmiennej wpisywana jest ilo wszystkich ukoczonych rozkazw
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_RequestQueueStats(POSNET_HANDLE hLocalDevice, unsigned long *sent, unsigned long *completed);	


	/*!
		\brief	Wyczy kolejk rozkazw oczekujcych na wysanie (nie ignoruj odpowiedzi na rozkazy ju wysane)

		Czyci kolejk rozkazw oczekujcych na wysanie. Funkcja powinna by wykorzystywana wycznie
		w sytuacji, gdy nastpi krytyczny bd urzdzenia. Rozkazy ju wysane zostan obsuone, jeli
		w dalszej perspektywie pojawi si na nie odpowied. Domylnie rozkazy czyszczone s wycznie dla biecego
		kontekstu urzdzenia.

		\param	hLocalDevice	Uchwyt lokalny urzdzenia
		\param globalnie Jeli parametr ten jest rny od 0 ignorowany jest kontekst urzdzenia.
	*/
	POSNET_API	POSNET_STATUS	__stdcall POS_RequestQueueCancelPending(POSNET_HANDLE hLocalDevice,unsigned char globalnie=0);
	/*!
		\brief	Wyczy kolejk rozkazw oczekujcych na wykonanie (zarwno wysanych jak i oczekujcych na wysanie)

		Czyci kolejk rozkazw wychodzcych. Funkcja powinna by wykorzystywana wycznie
		w sytuacji, gdy nastpi krytyczny bd urzdzenia. Wszystkie otrzymane pniej
		wyniki dla rozkazw, ktre byy w trakcie wykonywania zostan odrzucone. 
		Domylnie rozkazy czyszczone s wycznie dla biecego
		kontekstu urzdzenia.

		\param	hLocalDevice	Uchwyt lokalny urzdzenia
		\param globalnie Jeli parametr ten jest rny od 0 ignorowany jest kontekst urzdzenia.
	*/
	POSNET_API  POSNET_STATUS	__stdcall POS_RequestQueueCancelAll(POSNET_HANDLE hLocalDevice,unsigned char globalnie=0);

	/*!
		\brief	Wysya do drukarki danie powtrzenia transmisji odpowiedzi

		Funkcja jest przydatna w sytuacji, gdy chcemy powtrzy odbir ramki jeli nastpi timeout i istnieje
		podejrzenie, e drukarka otrzymaa rozkaz ale odpowied do nas nie powrcia. Do powtrzenia uywana
		jest de facto sekwencja 'rpt'. Uniemoliwia to przypadkowe podwjne wykonanie sekwencji jeli wczeniej nie
		dotara do drukarki (zostanie zwrcony bd 13).
		Uycie moliwe jest jeli rozkaz znajduje si w stanie POSNET_RSTATE_SENT. 
		(por. \ref POS_GetRequestState)

		\param hRequest	Uchwyt rozkazu, ktry ma by ponowiony
		\return Status wykonania
	*/
	POSNET_API      POSNET_STATUS  __stdcall POS_RepeatRequest(POSNET_HANDLE hRequest);

	/*!
		@}
	*/

	// --------------------------------------------------------------------------------------------------------
	/*!
	\defgroup	sync_api	Obsuga synchroniczna drukarki
	@{
	*/
	
	// PLACEHOLDER

	/*!
		@}
	*/
	// --------------------------------------------------------------------------------------------------------

}


#endif

